<?php
session_start();
include 'db.php';
if (!isset($_SESSION['user_id'])) { header("Location: login"); exit; }

$user_id = (int)$_SESSION['user_id'];
$user_name = $_SESSION['user_name'] ?? '';

$start_product = isset($_GET['start']) ? (int)$_GET['start'] : 0;
$selected_cid = isset($_GET['cid']) ? (int)$_GET['cid'] : 0;

// --- Determine product_id and receiver_id ---
$product_id = 0;
$receiver_id = 0;

if ($selected_cid) {
    // Existing conversation
    $stmt = $conn->prepare("SELECT product_id, sender_id, receiver_id FROM product_conversations WHERE conversation_id=? LIMIT 1");
    $stmt->bind_param("i", $selected_cid);
    $stmt->execute();
    $conv = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    if ($conv) {
        $product_id = (int)$conv['product_id'];
        $receiver_id = ($user_id == $conv['sender_id']) ? (int)$conv['receiver_id'] : (int)$conv['sender_id'];
    }
} elseif ($start_product) {
    // New conversation from product
    $product_id = $start_product;
    $stmt = $conn->prepare("SELECT seller_id FROM products WHERE id=? LIMIT 1");
    $stmt->bind_param("i", $start_product);
    $stmt->execute();
    $pr = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    if ($pr) $receiver_id = (int)$pr['seller_id'];
}

// --- Fetch conversations ---
$sql = "
SELECT pc.conversation_id,
       pc.product_id,
       p.name AS product_name,
       p.image_url,
       MAX(pc.message_id) AS last_msg_id,
       SUBSTRING_INDEX(GROUP_CONCAT(pc.message ORDER BY pc.message_id DESC SEPARATOR '|||'), '|||', 1) AS last_message,
       MAX(pc.created_at) AS last_time
FROM product_conversations pc
LEFT JOIN products p ON p.id = pc.product_id
WHERE pc.conversation_id > 0
  AND (pc.sender_id = ? OR pc.receiver_id = ?)
GROUP BY pc.conversation_id, pc.product_id
ORDER BY last_time DESC
";
$stmt = $conn->prepare($sql);
$stmt->bind_param("ii", $user_id, $user_id);
$stmt->execute();
$res = $stmt->get_result();
$conversations = [];
while ($r = $res->fetch_assoc()) $conversations[] = $r;
$stmt->close();
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Messages</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
body { background:#f8f9fa; }
.convo-list { max-height:80vh; overflow:auto; }
.chat-thread { max-height:70vh; overflow:auto; background:#fff; padding:15px; border-radius:8px; display:flex; flex-direction:column; }
.bubble { padding:10px 12px; border-radius:8px; margin-bottom:8px; max-width:80%; }
.bubble.me { background:#e1f0ff; align-self:flex-end; }
.bubble.other { background:#f1f1f1; align-self:flex-start; }
.thread-row { display:flex; flex-direction:column; }
.thread-row.me { align-items:flex-end; }
.convo-item img { width:48px; height:48px; object-fit:cover; border-radius:6px; margin-right:10px; }
.small-muted { font-size:0.85rem; color:#666; }
@media (max-width:768px){
    .row { flex-direction:column; }
    .col-md-4, .col-md-8 { flex:1; }
    .chat-thread { max-height:50vh; }
}
.toast-container { position: fixed; top: 20px; right: 20px; z-index: 9999; }
</style>
</head>
<body>
<div class="container py-3">
    <h4 class="mb-3">Messages</h4>
    <div class="row">
        <div class="col-md-4 mb-2">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <div>Conversations</div>
                </div>
                <div class="list-group convo-list list-group-flush">
                    <?php if (empty($conversations)): ?>
                        <div class="p-3 text-muted">No conversations yet.</div>
                    <?php endif; ?>
                    <?php foreach ($conversations as $c): ?>
                        <a href="?cid=<?= (int)$c['conversation_id'] ?>" class="list-group-item list-group-item-action <?= ($selected_cid && (int)$selected_cid === (int)$c['conversation_id']) ? 'active' : '' ?> d-flex align-items-center convo-item">
                            <img src="<?= htmlspecialchars($c['image_url'] ?: 'uploads/default.png') ?>" alt="">
                            <div class="flex-fill">
                                <div class="d-flex justify-content-between">
                                    <strong><?= htmlspecialchars($c['product_name'] ?: 'Product #'.(int)$c['product_id']) ?></strong>
                                    <small class="small-muted"><?= date('d M H:i', strtotime($c['last_time'])) ?></small>
                                </div>
                                <div class="text-truncate small"><?= htmlspecialchars($c['last_message']) ?></div>
                            </div>
                        </a>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <div class="col-md-8 d-flex flex-column">
            <div id="chatArea" class="chat-thread mb-2"></div>

            <form id="chatForm" class="d-flex">
                <input type="hidden" name="conversation_id" value="<?= (int)$selected_cid ?>">
                <input type="hidden" name="product_id" value="<?= (int)$product_id ?>">
                <input type="hidden" name="receiver_id" value="<?= (int)$receiver_id ?>">
                <input type="text" name="message" class="form-control me-2" placeholder="Type a message..." required>
                <button class="btn btn-primary">Send</button>
            </form>
        </div>
    </div>
</div>

<!-- Toast container -->
<div class="toast-container" id="toast-container"></div>

<script>
const userId = <?= json_encode($user_id) ?>;
let cid = <?= json_encode($selected_cid) ?>;
const pollInterval = 3000;

async function fetchMessages() {
    const chatArea = document.getElementById('chatArea');
    if (!cid) {
        chatArea.innerHTML = '<div class="text-muted">Select a conversation or start from a product.</div>';
        return;
    }
    try {
        const res = await fetch('fetch_messages.php?conversation_id=' + cid);
        if (!res.ok) return;
        const data = await res.json();
        chatArea.innerHTML = '';
        data.forEach(m => {
            const row = document.createElement('div');
            row.className = 'thread-row ' + (m.sender_id==userId?'me':'');
            const bubble = document.createElement('div');
            bubble.className = 'bubble ' + (m.sender_id==userId?'me':'other');
            bubble.innerHTML = '<div class="small text-muted">'+(m.sender_id==userId?'You':'Other')+' • '+m.created_at+'</div><div>'+m.message.replace(/\n/g,'<br>')+'</div>';
            row.appendChild(bubble);
            chatArea.appendChild(row);
        });
        chatArea.scrollTop = chatArea.scrollHeight;
    } catch(e){}
}

setInterval(fetchMessages, pollInterval);
fetchMessages();

document.getElementById('chatForm').addEventListener('submit', async function(e){
    e.preventDefault();
    const fd = new FormData(this);
    const res = await fetch('send_message.php', { method:'POST', body:fd });
    const json = await res.json();
    if (json.error) { showToast(json.error); return; }
    if (json.conversation_id) {
        cid = json.conversation_id;
        this.querySelector('[name="conversation_id"]').value = cid;
        // reload page to update conversation list
        window.location = 'messages.php?cid=' + cid;
    } else {
        fetchMessages();
    }
});

function showToast(msg){
    let toast = document.createElement('div');
    toast.className = 'toast align-items-center text-bg-primary border-0 show mb-2';
    toast.setAttribute('role','alert');
    toast.innerHTML = `<div class="d-flex"><div class="toast-body">${msg}</div>
        <button type="button" class="btn-close btn-close-white me-2 m-auto" onclick="this.parentElement.parentElement.remove()"></button></div>`;
    document.getElementById('toast-container').appendChild(toast);
    setTimeout(()=>toast.remove(),3000);
}
</script>
</body>
</html>
